 <!-- <?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Http\Resources\OrderplacedResource;
use App\Http\Resources\WishlistResource;
use App\Models\OrderDetails;
use Illuminate\Http\Request;
use App\Models\Wishlist;
use App\Models\Product;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class WishlistController extends Controller
{
      public function wishlist_delete($product_id)
    {
        $wishlist = Wishlist::where('product_id', $product_id)->first();
        if ($wishlist) {
            $wishlist->delete();
            return response()->json([
                'status'  => 'success',
                'message' => 'Product removed from wishlist',
                'data'    => WishlistResource::collection($wishlist),
                'code'    => 200,
            ]);
        }
        return response()->json([
            'status'  => 'error',
            'message' => 'Error not found',
            'data'    => WishlistResource::collection($wishlist),
            'code'    => 404,
        ]);
    }

    public function wishlist_display()
    {
        $user          = Auth::user();
        $wishlistItems = Wishlist::all();
        return response()->json([
            'status' => 'success',
            'data' => WishlistResource::collection($wishlistItems),
            'message' => 'Wishlist items retrieved successfully',
            'code' => 200
        ], 200);
    }
    public function wishlist_create(Request $request)
    {
        try {
            if (!auth()->guard('sanctum')->check()) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Please login!',
                    'data'    => [],
                    'code'    => 401,
                ]);
            }

            $user = auth()->guard('sanctum')->user();
            $productId = $request->input('product_id');
            if (is_null($productId)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Product ID is required',
                    'data' => [],
                    'code' => 400,
                ]);
            }


            $product = Product::find($productId);
            if (!$product) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Product not found',
                    'data'    => [],
                    'code'    => 404,
                ]);
            }


            $existingWishlistItem = Wishlist::where('user_id', $user->id)
                ->where('product_id', $productId)
                ->first();
            if ($existingWishlistItem) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Product is already in your wishlist',
                    'data'    => [],
                    'code'    => 409,
                ]);
            }


            $wishlistItem = new Wishlist([
                'user_id'      => $user->id,
                'product_id'   => $productId,
                'product_name' => $product->title
            ]);
            $wishlistItem->save();

            return response()->json([
                'status'  => 'success',
                'message' => 'Product added to wishlist',
                'data'    => [
                    'wishlist_id'  => $wishlistItem->id,
                    'product_id'   => $wishlistItem->product_id,
                    'product_name' => $wishlistItem->product_name,
                ],
                'code' => 201,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to add product to wishlist', ['error' => $e->getMessage()]);
            return response()->json([
                'status'  => 'error',
                'message' => 'Failed to add product to wishlist',
                'data'    => [],
                'code'    => 500,
            ]);
        }
     }

public function order_place(Request $request)
{
    if (!auth()->guard('sanctum')->check()) {
        return response()->json([
            'status'  => 'error',
            'message' => 'Please login!',
            'data'    => [],
            'code'    => 401,
        ]);
    }

    $user = auth()->guard('sanctum')->user();
    $validated = $request->validate([
        'product_details_id.*'   => 'required|string',
        'product_details_name.*' => 'required|string',
        'quantity.*'             => 'required|min:1',
        'amount.*'               => 'required|min:0',
        'delivery_address'       => 'required|string',
        'instruction'            => 'nullable|string',
        'category_id'            => 'nullable|string',
        'category_name'          => 'nullable|string',
    ]);

    $idsArray        = $validated['product_details_id'];
    $namesArray      = $validated['product_details_name'];
    $quantitiesArray = $validated['quantity'];
    $amountsArray    = $validated['amount'];

    $count = count($idsArray);
    if ($count !== count($namesArray) || $count !== count($quantitiesArray) || $count !== count($amountsArray)) {
        return response()->json([
            'status'  => 'error',
            'message' => 'Input arrays do not match in length.',
            'code'    => 400,
        ], 400);
    }

    $totalAmount = 0;
    foreach ($amountsArray as $index => $amount) {
        $quantity     = intval($quantitiesArray[$index]);
        $amount       = floatval($amount);
        $totalAmount += $amount * $quantity;
    }

    $idsString        = implode('~', $idsArray);
    $namesString      = implode('~', $namesArray);
    $quantitiesString = json_encode($quantitiesArray); 
    $amountString     = json_encode($amountsArray);

    $order = OrderDetails::create([
        'order_date'           => now(),
        'user_id'              => $user->id, 
        'user_name'            => $user->name, 
        'category_id'          => $validated['category_id'] ?? null,
        'category_name'        => $validated['category_name'] ?? null,
        'product_details_id'   => $idsString,
        'product_details_name' => $namesString,
        'quantity'             => $quantitiesString,
        'amount'               => number_format($totalAmount, 2), 
        'delivery_address'     => $validated['delivery_address'],
        'instruction'          => $validated['instruction'] ?? ''
    ]);

    return response()->json([
        'status'  => 'success',
        'message' => 'Order placed successfully',
        'data'    => new OrderplacedResource($order),
        'code'    => 200,
    ]);
}

} 


 