<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Http\Resources\DeliveryFeeResource;
use App\Http\Resources\OrderplacedResource;
use App\Http\Resources\SearchProductResources;
use App\Http\Resources\WishlistResource;
use App\Models\Address;
use App\Models\DeliveryFeeMaster;
use Illuminate\Support\Facades\DB;
use App\Models\OrderDetails;
use App\Models\OrderItem;
use App\Models\ProductVarient;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;
use App\Models\Wishlist;
use Illuminate\Support\Str;
use App\Models\Product;
use App\Models\Cart;
use App\Models\Order;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Laravel\Prompts\Themes\Default\SearchPromptRenderer;

class WishlistController extends Controller
{

    
public function wishlist_display()
{
    try {
        $user = Auth::user();
        if (!$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'User not authenticated!',
                'data' => [],
                'code' => 401,
            ], 401);
        }
        $wishlistItems = Wishlist::with(['product.variants'])
            ->where('user_id', $user->id)
            ->get();
        $wishlistItemsWithDetails = $wishlistItems->map(function ($wishlistItem) {
            $product = $wishlistItem->product;
            if ($product) {
                $product->setAttribute('is_wishlist', true);
                $productDetails = [
                    'id'          => $product->id,
                    'title'       => $product->title,
                    'description' => $product->description,
                    'image'       => $product->image,
                    'variants'    => $product->variants->map(function ($variant) {
                        $mrp = number_format($variant->mrp, 2);
                        $discountedPrice = $variant->discount ? number_format($variant->mrp - ($variant->mrp * ($variant->discount / 100)), 2) : null;
                        $discountPercentage = $variant->discount ? $variant->discount : 0;
                        return [
                            'id'       => $variant->id,
                            'title'    => $variant->title,
                            'mrp'      => $mrp,
                            'discount' => $variant->discount,
                            'price'    => $discountedPrice,
                           
                        ];
                    }),
                ];

                return [
                    'id'      => $wishlistItem->id,
                    'user_id' => $wishlistItem->user_id,
                    'product' => $productDetails,
                ];
            }

            return [];
        })->filter(); 

        return response()->json([
            'status'  => 'success',
            'data'    => $wishlistItemsWithDetails,
            'message' => 'Wishlist items retrieved successfully',
            'code'    => 200
        ], 200);
    } catch (\Exception $e) {
        return response()->json([
            'status'  => 'error',
            'message' => 'Failed to retrieve wishlist items.',
            'code'    => 500,
        ], 500);
    }
}


    public function wishlist(Request $request)
    {
        try {
            if (!auth()->guard('sanctum')->check()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Please login!',
                    'data' => [],
                    'code' => 401,
                ]);
            }
            $user = auth()->guard('sanctum')->user();
            $productId = $request->input('product_id');
            if (is_null($productId)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Product ID is required',
                    'data' => [],
                    'code' => 400,
                ]);
            }
            $product = Product::find($productId);
            if (!$product) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Product not found',
                    'data' => [],
                    'code' => 404,
                ]);
            }
            $existingWishlistItem = Wishlist::where('user_id', $user->id)
                ->where('product_id', $productId)
                ->first();
            if ($existingWishlistItem) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Product is already in your wishlist',
                    'data' => [],
                    'code' => 409,
                ]);
            }
            $wishlistItem = new Wishlist([
                'user_id' => $user->id,
                'product_id' => $productId,
                'product_name' => $product->title,
            ]);
            $wishlistItem->save();
            // Update the product's is_wishlisted column
            $product->update(['is_wishlisted' => 1]);
            return response()->json([
                'status' => 'success',
                'message' => 'Product added to wishlist',
                'data' => [
                    'user_id' => $wishlistItem->user_id,
                    'wishlist_id' => $wishlistItem->id,
                    'product_id' => $wishlistItem->product_id,
                    'product_name' => $wishlistItem->product_name,
                ],
                'code' => 201,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to add product to wishlist', ['error' => $e->getMessage()]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to add product to wishlist',
                'data' => [],
                'code' => 500,
            ]);
        }
    }
    public function wishlist_delete($product_id)
    {
        $wishlist = Wishlist::where('product_id', $product_id)->first();
        if ($wishlist) {
            $wishlist->delete();
            // Update the product's is_wishlisted column
            $product = Product::find($product_id);
            $product->update(['is_wishlisted' => 0]);
            return response()->json([
                'status' => 'success',
                'message' => 'Product removed from wishlist',
                'data' => [],
                'code' => 200,
            ]);
        }
        return response()->json([
            'status' => 'error',
            'message' => 'Error not found',
            'data' => [],
            'code' => 404,
        ]);
    }
    public function wishlist_update(Request $request)
    {
        try {
            // Check if the user is authenticated
            if (!auth()->guard('sanctum')->check()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Please login!',
                    'data' => [],
                    'code' => 401,
                ]);
            }
            $user = auth()->guard('sanctum')->user();
            $productId = $request->input('product_id');
            // Validate the product ID
            if (is_null($productId)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Product ID is required',
                    'data' => [],
                    'code' => 400,
                ]);
            }
            // Find the product
            $product = Product::find($productId);
            if (!$product) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Product not found',
                    'data' => [],
                    'code' => 404,
                ]);
            }
            // Check if the product is already in the user's wishlist
            $existingWishlistItem = Wishlist::where('user_id', $user->id)
                ->where('product_id', $productId)
                ->first();
            if ($existingWishlistItem) {
                // If the product is in the wishlist, remove it
                $existingWishlistItem->delete();
                // Optionally update the product's is_wishlisted column (if needed)
                // $product->update(['is_wishlisted' => 0]);
                return response()->json([
                    'status' => 'success',
                    'message' => 'Product removed from wishlist',
                    'data' => [
                        'user_id' => $user->id,
                        'product_id' => $productId,
                        'product_name' => $product->title,
                    ],
                    'code' => 200,
                ]);
            } else {
                // If the product is not in the wishlist, add it
                $wishlistItem = new Wishlist([
                    'user_id' => $user->id,
                    'product_id' => $productId,
                    'product_name' => $product->title,
                ]);
                $wishlistItem->save();
                // Optionally update the product's is_wishlisted column (if needed)
                // $product->update(['is_wishlisted' => 1]);
                return response()->json([
                    'status' => 'success',
                    'message' => 'Product added to wishlist',
                    'data' => [
                        'user_id'      => $wishlistItem->user_id,
                        'wishlist_id'  => $wishlistItem->id,
                        'product_id'   => $wishlistItem->product_id,
                        'product_name' => $wishlistItem->product_name,
                    ],
                    'code' => 201,
                ]);
            }
        } catch (\Exception $e) {
            // Log the error for debugging
            Log::error('Failed to update wishlist', ['error' => $e->getMessage()]);

            return response()->json([
                'status'  => 'error',
                'message' => 'Failed to update wishlist',
                'data'    => [],
                'code'    => 500,
            ]);
        }
    }

   
    public function searchProduct(Request $request)
    {
        $searchTerm = $request->input('search_term');
        if (!$searchTerm) {
            return response()->json([
                'status' => 'error',
                'message' => 'Please provide a search term.',
                'code' => 422,
            ], 422);
        }
    
        try {
            $products = Product::with(['variants'])
                ->where('title', 'LIKE', "{$searchTerm}%")
                ->get();
    
            if ($products->isEmpty()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No products found.',
                    'code' => 404,
                ], 404);
            }
    
            return response()->json([
                'status' => 'success',
                'data' => SearchProductResources::collection($products),
                'message' => 'Products found.',
                'code' => 200,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred while searching for products.',
                'error' => $e->getMessage(),
                'code' => 500,
            ], 500);
        }
    }
    
    
    
   


}